// Registry.cpp : Implementation of CRegistry

#include "stdafx.h"
#include "MGCEWin32.h"
#include "Registry.h"

/////////////////////////////////////////////////////////////////////////////
// CRegistry


STDMETHODIMP CRegistry::get_Key(long *pVal)
{
	*pVal = (DWORD)hKey & 0x0000FFFF;
	return S_OK;
}

STDMETHODIMP CRegistry::put_Key(long newVal)
{
	hKey = (HKEY)( (DWORD)newVal + 0x80000000 );
	return S_OK;
}

STDMETHODIMP CRegistry::get_Path(BSTR *pVal)
{
	if( ! bPath )  *pVal = SysAllocString( L"\0" );
	else *pVal = SysAllocString( bPath );
	return S_OK;
}

STDMETHODIMP CRegistry::put_Path(BSTR newVal)
{
	if( bPath ) SysFreeString( bPath );
	if( ! wcslen( newVal ) )  bPath = NULL;
	else bPath = SysAllocString( newVal );
	return S_OK;
}

STDMETHODIMP CRegistry::get_Name(BSTR *pVal)
{
	if( ! bName ) *pVal = SysAllocString( L"\0" );
	else *pVal = SysAllocString( bName );
	return S_OK;
}

STDMETHODIMP CRegistry::put_Name(BSTR newVal)
{
	if( bName ) SysFreeString( bName );
	bName = SysAllocString( newVal );
	return S_OK;
}

STDMETHODIMP CRegistry::get_ValueType(long *pVal)
{
	LONG lResult;
	HKEY hNewKey;

	if( ! bPath || ! bName ) {
		Error( L"No path or item name specified" );
		return E_FAIL;
	}
	lResult = ::RegOpenKeyEx( hKey, bPath, 0, 0, &hNewKey );
	if( lResult == ERROR_SUCCESS ) {
		::RegQueryValueEx( hNewKey, bName, NULL, (LPDWORD)pVal, NULL, NULL );
		::RegCloseKey( hNewKey );
	}
	if( lResult != ERROR_SUCCESS ) {
		Error( L"Error retriving value type" );
		return E_FAIL;
	}
	return S_OK;
}

STDMETHODIMP CRegistry::put_ValueType(long newVal)
{
	lType = newVal;
	return S_OK;
}

STDMETHODIMP CRegistry::CreatePath()
{
	LONG lResult;
	HKEY hNewKey;
	DWORD dwDisp;

	if( ! bPath ) {
		Error( L"No parent path was specified" );
		return E_FAIL;
	}
	lResult = ::RegCreateKeyEx( hKey, bPath, 0, NULL, 0, 0, NULL, &hNewKey, &dwDisp );
	if( lResult == ERROR_SUCCESS ) {
		RegCloseKey( hNewKey );
		return S_OK;
	}
	Error( L"Error creating new path" );
	return E_FAIL;
}

STDMETHODIMP CRegistry::DeletePath()
{
	LONG lResult;

	if( ! bPath ) {
		Error( L"No path specified" );
		return E_FAIL;
	}
	lResult = ::RegDeleteKey( hKey, bPath );
	if( lResult == ERROR_SUCCESS ) return S_OK;
	Error( L"Error deleting specified path" );
	return E_FAIL;
}

STDMETHODIMP CRegistry::get_Value(VARIANT *pVal)
{
	LONG lResult;
	void *pPtr;
	DWORD dwSize, dwType, dwLen;
	TCHAR *pszOut, szCh[ 4 ];
	HKEY hNewKey;

	if( ! bPath ) {
		Error( L"No path specified" );
		return E_FAIL;
	}
	lResult = ::RegOpenKeyEx( hKey, bPath, 0, 0, &hNewKey );
	if( lResult != ERROR_SUCCESS ) {
		Error( L"Unable to open specified path" );
		return E_FAIL;
	}
	lResult = ::RegQueryValueEx( hNewKey, bName, NULL, &dwType, NULL, &dwSize );
	if( lResult != ERROR_SUCCESS ) {
		::RegCloseKey( hNewKey );
		Error( L"Error reading value from key" );
		return E_FAIL;
	}
	
	if( dwSize ) {
		pPtr = malloc( dwSize );
		if( ! pPtr ) {
			::RegCloseKey( hNewKey );
			return E_OUTOFMEMORY;
		}
	} else pPtr = NULL;

	if( dwSize ) {
		lResult = ::RegQueryValueEx( hNewKey, bName, NULL, &dwType,
									 (LPBYTE)pPtr, &dwSize );
		if( lResult != ERROR_SUCCESS ) {
			::RegCloseKey( hNewKey );
			free( pPtr );
			Error( L"Error reading value from key" );
			return E_FAIL;
		}
	}

	if( dwType == 1 || dwType == 2 || dwType == 6 ) {
		V_VT( pVal ) = VT_BSTR;
		if( dwSize )  V_BSTR( pVal ) = SysAllocString( (BSTR)pPtr );
		else          V_BSTR( pVal ) = SysAllocString( L"\0" );
	}
	else
	if( dwType == 4 || dwType == 5 ) {
		V_VT( pVal ) = VT_UI4;
		if( dwSize )  V_UI4( pVal ) = *(LPDWORD)pPtr;
		else          V_UI4( pVal ) = 0;
	}
	else
	if( dwType == 3 ) {
		if( dwSize ) {
			dwLen = ( dwSize + 1 ) * 2;
			pszOut = (TCHAR *)malloc( dwLen );
			if( ! pszOut ) {
				::RegCloseKey( hNewKey );
				free( pPtr );
				return E_OUTOFMEMORY;
			}
			*pszOut = L'\0';
			for( dwLen = 0; dwLen < dwSize; dwLen++ ) {
				wsprintf( szCh, L"%.2x", (BYTE)*( (LPBYTE)pPtr + dwLen ) );
				wcscat( pszOut, szCh );
			}
			_wcsupr( pszOut );
			V_BSTR( pVal ) = SysAllocString( pszOut );
			free( pszOut );
			V_VT( pVal ) = VT_BSTR;
		} else V_BSTR( pVal ) = SysAllocString( L"\0" );
	}
	else
	if( dwType == 7 ) {
		SAFEARRAY *psa;
		TCHAR *pszStr;
		int i, j;
		VARIANT *pV;

		pszStr = (TCHAR *)pPtr;
		i = 0;
		while( *pszStr ) {
			++i;
			pszStr = pszStr + wcslen( pszStr ) + 1;
		}
		psa = SafeArrayCreateVector( VT_VARIANT, 1, i );
		pszStr = (TCHAR *)pPtr;
		pV = (VARIANT *)psa->pvData;
		for( j = 0; j < i; j++ ) {
			V_BSTR( pV ) = SysAllocString( pszStr );
			V_VT( pV ) = VT_BSTR;
			pszStr = pszStr + wcslen( pszStr ) + 1;
			++pV;
		}
		V_VT( pVal ) = VT_ARRAY | VT_VARIANT;
		V_ARRAY( pVal ) = psa;
	}
	else V_VT( pVal ) = VT_EMPTY;

	free( pPtr );
	::RegCloseKey( hNewKey );
	return S_OK;
}

STDMETHODIMP CRegistry::put_Value(VARIANT *newVal)
{
	BOOL bStatus = FALSE;

	if( ! bPath || ! bName ) {
		Error( L"A path or name was not specified" );
		return E_FAIL;
	}
	if( lType == REG_SZ )        bStatus = SaveSZ( newVal );
	if( lType == REG_DWORD )     bStatus = SaveDWORD( newVal );
	if( lType == REG_MULTI_SZ )	 bStatus = SaveMultiSZ( newVal );
	if( lType == REG_BINARY )    bStatus = SaveBinary( newVal );
	if( bStatus == TRUE ) return S_OK;
	return E_FAIL;
}

//  Save variant contents as REG_SZ type

BOOL CRegistry::SaveSZ(VARIANT *pVar)
{
	LONG lResult;
	DWORD dwDisp;
	VARIANT varNew;
	HKEY hNewKey;
	HRESULT hr;

	VariantInit( &varNew );
	hr = VariantChangeType( &varNew, pVar, 0, VT_BSTR );
	if( FAILED( hr ) ) {
		Error( L"Unable to convert value to text string" );
		return FALSE;
	}

	lResult = ::RegCreateKeyEx( hKey, bPath, 0, NULL, 0, 0, 0, &hNewKey, &dwDisp );
	if( lResult != ERROR_SUCCESS ) {
		Error( L"Unable to open or create path" );
		return FALSE;
	}
	lResult |= ::RegSetValueEx( hNewKey, bName, 0, REG_SZ, (LPBYTE)varNew.bstrVal,
								( wcslen( varNew.bstrVal ) + 1 ) * sizeof(TCHAR) );
	
	SysFreeString( varNew.bstrVal );
	::RegCloseKey( hNewKey );
	if( lResult != ERROR_SUCCESS ) {
		Error( L"Error setting value" );
		return FALSE;
	}
	return TRUE;
}

// Save variant contents as REG_DWORD type

BOOL CRegistry::SaveDWORD(VARIANT *pVar)
{
	LONG lResult;
	DWORD dwDisp, dwVal;
	VARIANT varNew;
	HKEY hNewKey;
	HRESULT hr;

	if( pVar->vt == VT_R8 ) dwVal = (DWORD)V_R8( pVar );
	else {
		VariantInit( &varNew );
		hr = VariantChangeType( &varNew, pVar, 0, VT_UI4 );
		if( FAILED( hr ) ) {
			Error( L"Error converting data to DWORD value" );
			return FALSE;
		}
		dwVal = varNew.lVal;
	}
	lResult = ::RegCreateKeyEx( hKey, bPath, 0, NULL, 0, 0, 0, &hNewKey, &dwDisp );
	if( lResult != ERROR_SUCCESS ) {
		Error( L"Error opening or creating key" );
		return FALSE;
	}
	lResult |= ::RegSetValueEx( hNewKey, bName, 0, REG_DWORD, (LPBYTE)&dwVal,
								sizeof(DWORD) );
	::RegCloseKey( hNewKey );
	if( lResult != ERROR_SUCCESS ) {
		Error( L"Error writing data to registry" );
		return FALSE;
	}
	return TRUE;
}

//  Save array of variants in a single REG_MULTI_SZ type

BOOL CRegistry::SaveMultiSZ(VARIANT *pVar)
{
	LONG lResult;
	DWORD dwDisp, dwLBound, dwUBound, dwLen;
	int iElements, i;
	HKEY hNewKey;
	HRESULT hr;
	VARIANT *pArray, newVar;
	TCHAR **szOut, *pszTotal;

	if( ( pVar->vt & VT_ARRAY ) == 0 )  return( SaveSZ( pVar ) );

	SafeArrayAccessData( V_ARRAY( pVar ), (void **)&pArray );
	SafeArrayGetLBound( V_ARRAY( pVar ), 1, (long *)&dwLBound );
	SafeArrayGetUBound( V_ARRAY( pVar ), 1, (long *)&dwUBound );
	iElements = (int)dwUBound - (int)dwLBound + 1;
	if( ! iElements ) {
		SafeArrayUnaccessData( V_ARRAY( pVar ) );
		SafeArrayDestroy( V_ARRAY( pVar ) );
		Error( L"No elements in specified array" );
		return FALSE;
	}

	szOut = (TCHAR **)malloc( iElements * sizeof(TCHAR *) );
	if( ! szOut ) {
		SafeArrayUnaccessData( V_ARRAY( pVar ) );
		SafeArrayDestroy( V_ARRAY( pVar ) );
		return FALSE;
	}

	for( i = 0; i < iElements; i++ ) {
		VariantInit( &newVar );
		hr = VariantChangeType( &newVar, &pArray[ i ], 0, VT_BSTR );
		if( FAILED( hr ) )  break;
		szOut[ i ] = (TCHAR *)malloc( ( wcslen( newVar.bstrVal ) + 1 ) * sizeof(TCHAR) );
		if( ! szOut[ i ] ) break;
		wcscpy( szOut[ i ], newVar.bstrVal );
		SysFreeString( newVar.bstrVal );
	}
	
	if( i != iElements ) {
		for( i = 0; i < iElements; i++ )
			if( szOut[ i ] )  free( szOut[ i ] );
		free( szOut );
		SafeArrayUnaccessData( V_ARRAY( pVar ) );
		SafeArrayDestroy( V_ARRAY( pVar ) );
		Error( L"Unable to access or convert element to string" );
		return FALSE;
	}

	dwLen = 0;
	for( i = 0; i < iElements; i++ )
		dwLen = dwLen + (DWORD)wcslen( szOut[ i ] ) + 1;
	++dwLen;

	pszTotal = (TCHAR *)malloc( dwLen * sizeof(TCHAR) );
	if( ! pszTotal )
	{
		for( i = 0; i < iElements; i++ )
			if( szOut[ i ] )  free( szOut[ i ] );
		free( szOut );
		SafeArrayUnaccessData( V_ARRAY( pVar ) );
		SafeArrayDestroy( V_ARRAY( pVar ) );
		return FALSE;
	}
	memset( pszTotal, 0, dwLen * sizeof(TCHAR) );

	dwLen = 0;
	for( i = 0; i < iElements; i++ ) {
		wcscpy( pszTotal + dwLen, szOut[ i ] );
		dwLen = dwLen + wcslen( szOut[ i ] ) + 1;
	}
	++dwLen;

	lResult = ::RegCreateKeyEx( hKey, bPath, 0, NULL, 0, 0, 0, &hNewKey, &dwDisp );
	if( lResult != ERROR_SUCCESS ) return FALSE;
	lResult |= ::RegSetValueEx( hNewKey, bName, 0, REG_MULTI_SZ, (LPBYTE)pszTotal,
								dwLen * sizeof(TCHAR) );
	::RegCloseKey( hNewKey );
	for( i = 0; i < iElements; i++ )
		if( szOut[ i ] )  free( szOut[ i ] );
	free( szOut );
	SafeArrayUnaccessData( V_ARRAY( pVar ) );
	SafeArrayDestroy( V_ARRAY( pVar ) );
	return TRUE;
}

BOOL CRegistry::SaveBinary(VARIANT *pVar)
{
	LPBYTE pbData;
	BYTE bHigh, bLow;
	BSTR bData;
	DWORD dwLen, dwCount, dwDisp;
	HKEY hNewKey;
	LONG lResult;

	bData = V_BSTR( pVar );
	_wcsupr( bData );
	dwLen = wcslen( bData ) / 2;
	pbData = (LPBYTE)malloc( dwLen );
	if( ! pbData )  return FALSE;

	for( dwCount = 0; dwCount < dwLen; dwCount++ ) {
		bHigh = (BYTE)*( bData + ( dwCount * 2 ) );
		bLow = (BYTE)*( bData + ( dwCount * 2 ) + 1 );
		bHigh -= 48;
		bLow -=48;
		if( bHigh > 9 ) bHigh -= 7;
		if( bLow > 9 ) bLow -= 7;
		*( pbData + dwCount ) = ( bHigh * 16 ) + bLow;
	}

	lResult = ::RegCreateKeyEx( hKey, bPath, 0, NULL, 0, 0, 0, &hNewKey, &dwDisp );
	if( lResult != ERROR_SUCCESS ) {
		free( pbData );
		return FALSE;
	}
	lResult |= ::RegSetValueEx( hNewKey, bName, 0, REG_BINARY, pbData, dwLen );
	::RegCloseKey( hNewKey );
	free( pbData );
	return( TRUE );
}

STDMETHODIMP CRegistry::DeleteValue()
{
	LONG lResult;
	HKEY hNewKey;

	if( ! bPath || ! bName ) {
		Error( L"No path or name specified" );
		return E_FAIL;
	}
	lResult = ::RegOpenKeyEx( hKey, bPath, 0, 0, &hNewKey );
	if( lResult != ERROR_SUCCESS ) {
		Error( L"Unable to open path" );
		return E_FAIL;
	}
	lResult = ::RegDeleteValue( hNewKey, bName );
	::RegCloseKey( hNewKey );
	if( lResult != ERROR_SUCCESS ) {
		Error( L"Error deleting specified item" );
		return E_FAIL;
	}
	return S_OK;
}

STDMETHODIMP CRegistry::GetPathName(long lVal, BSTR *pRet)
{
	LONG lResult;
	HKEY hNewKey;
	TCHAR szName[ 512 ];
	DWORD dwSize;

	lResult = ERROR_SUCCESS;
	if( bPath )	lResult = ::RegOpenKeyEx( hKey, bPath, 0, 0, &hNewKey );
	else hNewKey = hKey;
	if( lResult != ERROR_SUCCESS ) {
		Error( L"Unable to open path" );
		return E_FAIL;
	}
	dwSize = 512;
	lResult = ::RegEnumKeyEx( hNewKey, lVal, szName, &dwSize, NULL, NULL, NULL, NULL );
	if( lResult == ERROR_SUCCESS )  *pRet = SysAllocString( szName );
	if( bPath )  ::RegCloseKey( hNewKey );
	if( lResult == ERROR_SUCCESS )  return S_OK;
	return E_FAIL;
}

STDMETHODIMP CRegistry::GetValueName(long lVal, BSTR *pRet)
{
	LONG lResult;
	HKEY hNewKey;
	TCHAR szName[ 512 ];
	DWORD dwSize;

	lResult = ERROR_SUCCESS;
	if( bPath ) lResult = ::RegOpenKeyEx( hKey, bPath, 0, 0, &hNewKey );
	else hNewKey = hKey;
	if( lResult != ERROR_SUCCESS ) {
		Error( L"Unable to open path" );
		return E_FAIL;
	}
	dwSize = 512;
	lResult = ::RegEnumValue( hNewKey, lVal, szName, &dwSize, NULL, NULL, NULL, NULL );
	if( lResult == ERROR_SUCCESS )  *pRet = SysAllocString( szName );
	if( bPath )  ::RegCloseKey( hNewKey );
	if( lResult == ERROR_SUCCESS )  return S_OK;
	return E_FAIL;
}


STDMETHODIMP CRegistry::get_Version(long *pVal)
{
	*pVal = 101;
	return S_OK;
}
